<?php

namespace ThirstyAffiliates_Pro\Models;

use ThirstyAffiliates_Pro\Abstracts\Abstract_Main_Plugin_Class;

use ThirstyAffiliates_Pro\Interfaces\Model_Interface;

use ThirstyAffiliates_Pro\Helpers\Plugin_Constants;
use ThirstyAffiliates_Pro\Helpers\Helper_Functions;

/**
 * Model that houses the logic for Addons_Updates
 *
 */
class Addons_Updates implements Model_Interface {

  /*
  |--------------------------------------------------------------------------
  | Class Properties
  |--------------------------------------------------------------------------
  */

  /**
   * Model that houses all the plugin constants.
   *
   * @since 3.0.0
   * @access private
   * @var Plugin_Constants
   */
  private $_constants;

  /**
   * Property that houses all the helper functions of the plugin.
   *
   * @since 3.0.0
   * @access private
   * @var Helper_Functions
   */
  private $_helper_functions;

  /**
   * Property that houses all the helper functions of the plugin.
   *
   * @access private
   * @var Mothership\Update
   */
  private $_update;

  /**
   *
   * @access private
   * @var String
   */
  private $license_key;


  public $slug, $main_file, $installed_version, $name, $description;

  /*
  |--------------------------------------------------------------------------
  | Class Methods
  |--------------------------------------------------------------------------
  */

  /**
   * Class constructor.
   *
   * @since 3.0.0
   * @access public
   *
   * @param Plugin_Constants           $constants        Plugin constants object.
   * @param Helper_Functions           $helper_functions Helper functions object.
   */
  public function __construct( Plugin_Constants $constants , Helper_Functions $helper_functions ) {

      $this->_constants        = $constants;
      $this->_helper_functions = $helper_functions;
  }

  public function set_props($slug, $main_file, $installed_version, $name, $description){
    $this->slug                   = $slug;
    $this->main_file              = $main_file;
    $this->installed_version      = $installed_version;
    $this->name                   = $name;
    $this->description            = $description;

    $this->_update = ThirstyAffiliates_Pro()->get_model('Update');
    $this->license_key = $this->_helper_functions->get_option( Plugin_Constants::OPTION_LICENSE_KEY );

    return $this;
  }

  /**
   * Execute shortcodes class.
   *
   * @since 3.0.0
   * @access public
   */
  public function run() {
    if( is_object($this->_update) && ! empty($this->main_file) ) {
      $this->hooks();
    }
  }

  /**
  * Register hooks.
  */
  private function hooks() {
    add_filter('pre_set_site_transient_update_plugins', array($this, 'queue_update'));
    add_filter('plugins_api', array($this, 'plugin_info'), 11, 3);
    add_action("in_plugin_update_message-$this->main_file", array($this, 'check_incorrect_edition'));
    add_action('tap_plugin_edition_changed', array($this, 'clear_update_transient'));
    add_action('tap_license_activated_before_queue_update', array($this, 'clear_update_transient'));
    add_action('tap_license_deactivated_before_queue_update', array($this, 'clear_update_transient'));
  }

  public function queue_update($transient) {
    if(empty($transient->checked)) {
      return $transient;
    }

    $update_info = get_site_transient('tap_update_info_' . $this->slug);

    if(!is_array($update_info)) {
      $args = array();

      $edge_updates = $this->_helper_functions->get_option( Plugin_Constants::OPTION_EDGE_UPDATES );
      if($edge_updates || (defined('THIRSTYAFFILIATES_PRO_EDGE_UPDATES') && THIRSTYAFFILIATES_PRO_EDGE_UPDATES)) {
        $args['edge'] = 'true';
      }

      if(empty($this->license_key)) {
        // Just here to query for the current version
        try {
          $version_info = $this->_helper_functions->send_mothership_request("/versions/latest/{$this->slug}", $args);
          $curr_version = $version_info['version'];
          $download_url = '';
        }
        catch(\Exception $e) {
          if(isset($transient->response[$this->main_file])) {
            unset($transient->response[$this->main_file]);
          }

          return $transient;
        }
      }
      else {
        try {
          $helper_functions = ThirstyAffiliates()->helpers['Helper_Functions'];
          $args['domain'] = urlencode($helper_functions->site_domain());

          $license_info = $this->_helper_functions->send_mothership_request("/versions/info/{$this->slug}/{$this->license_key}", $args);
          $curr_version = $license_info['version'];
          $download_url = $license_info['url'];

          if($this->_helper_functions->is_incorrect_edition_installed()) {
            $download_url = '';
          }
        }
        catch(\Exception $e) {
          try {
            // Just here to query for the current version
            $version_info = $this->_helper_functions->send_mothership_request("/versions/latest/{$this->slug}", $args);
            $curr_version = $version_info['version'];
            $download_url = '';
          } catch (\Exception $e) {
            if(isset($transient->response[$this->main_file])) {
              unset($transient->response[$this->main_file]);
            }

            return $transient;
          }
        }
      }

      set_site_transient(
        'tap_update_info_' . $this->slug,
        compact('curr_version', 'download_url'),
        (12 * HOUR_IN_SECONDS)
      );
    }
    else {
      $curr_version = isset($update_info['curr_version']) ? $update_info['curr_version'] : $this->installed_version;
      $download_url = isset($update_info['download_url']) ? $update_info['download_url'] : '';
    }

    if(isset($curr_version) && version_compare($curr_version, $this->installed_version, '>')) {
      global $wp_version;

      $transient->response[$this->main_file] = (object) array(
        'id' => $this->main_file,
        'slug' => $this->slug,
        'plugin' => $this->main_file,
        'new_version' => $curr_version,
        'url' => 'https://thirstyaffiliates.com/',
        'package' => $download_url,
        'tested' => $wp_version
      );
    }
    else {
      unset($transient->response[$this->main_file]);

      // Enables the "Enable auto-updates" link
      $transient->no_update[$this->main_file] = (object) array(
        'id' => $this->main_file,
        'slug' => $this->slug,
        'plugin' => $this->main_file,
        'new_version' => $this->installed_version,
        'url' => 'https://thirstyaffiliates.com/',
        'package' => ''
      );
    }

    return $transient;
  }

  public function plugin_info($api, $action, $args) {
    global $wp_version;

    if(!isset($action) || $action != 'plugin_information') {
      return $api;
    }

    if(!isset($args->slug) || $args->slug != $this->slug) {
      return $api;
    }

    $args = array();

    $edge_updates = $this->_helper_functions->get_option( Plugin_Constants::OPTION_EDGE_UPDATES );

    if($edge_updates || (defined('THIRSTYAFFILIATES_PRO_EDGE_UPDATES') && THIRSTYAFFILIATES_PRO_EDGE_UPDATES)) {
      $args['edge'] = 'true';
    }

    if(empty($this->license_key)) {
      try {
        // Just here to query for the current version
        $version_info = $this->_helper_functions->send_mothership_request("/versions/latest/{$this->slug}", $args);
        $curr_version = $version_info['version'];
        $version_date = $version_info['version_date'];
        $download_url = '';
      }
      catch(\Exception $e) {
        return $api;
      }
    }
    else {
      try {
        $helper_functions = ThirstyAffiliates()->helpers['Helper_Functions'];
        $args['domain'] = urlencode($helper_functions->site_domain());

        $license_info = $this->_helper_functions->send_mothership_request("/versions/info/{$this->slug}/{$this->license_key}", $args);
        $curr_version = $license_info['version'];
        $version_date = $license_info['version_date'];
        $download_url = $license_info['url'];
      }
      catch(\Exception $e) {
        try {
          // Just here to query for the current version
          $version_info = $this->_helper_functions->send_mothership_request("/versions/latest/{$this->slug}", $args);
          $curr_version = $version_info['version'];
          $version_date = $version_info['version_date'];
          $download_url = '';
        }
        catch(\Exception $e) {
          return $api;
        }
      }
    }

    return (object) array(
      'slug' => $this->slug,
      'name' => esc_html($this->name),
      'author' => TAP_AUTHOR,
      'author_profile' => TAP_AUTHOR_URI,
      'contributors' => array(
        'caseproof' => array(
          'profile' => TAP_AUTHOR_URI,
          'avatar' => 'https://secure.gravatar.com/avatar/762b61e36276ff6dc0d7b03b8c19cfab?s=96&d=monsterid&r=g',
          'display_name' => TAP_AUTHOR
        )
      ),
      'homepage' => 'https://thirstyaffiliates.com/',
      'version' => $curr_version,
      'new_version' => $curr_version,
      'requires' => '5.2',
      'requires_php' => '5.2.0',
      'tested' => $wp_version,
      'compatibility' => array($wp_version => array($curr_version => array(100, 0, 0))),
      'rating' => '100.00',
      'num_ratings' => '1',
      'added' => '2012-12-02',
      'last_updated' => $version_date,
      'tags' => array(
        'affiliate links' => 'affiliate links',
        'link branding' => 'link branding',
        'link shortner' => 'link shortner',
        'link tracking' => 'link tracking',
        'redirects' => 'redirects'
      ),
      'sections' => array(
        'description' => '<p>' . $this->description . '</p>',
        'faq' => '<p>' . sprintf(esc_html__('You can access in-depth information about ThirstyAffiliates at %1$sthe ThirstyAffiliates User Manual%2$s.', 'thirstyaffiliates-pro'), '<a href="https://thirstyaffiliates.com/knowledge-base">', '</a>') . '</p>'
      ),
      'download_link' => $download_url
    );
  }

  public function check_incorrect_edition() {
    if($this->_helper_functions->is_incorrect_edition_installed()) {
      printf(
        /* translators: %1$s: open link tag, %2$s: close link tag */
        ' <strong>' . esc_html__('To restore automatic updates, %1$sinstall the correct edition%2$s of Thirsty Affiliates.', 'thirstyaffiliates-pro') . '</strong>',
        sprintf('<a href="%s">', esc_url(admin_url('edit.php?post_type='.\ThirstyAffiliates\Helpers\Plugin_Constants::AFFILIATE_LINKS_CPT.'&page=thirsty-settings&tab=tap_mothership_settings_section'))),
        '</a>'
      );
    }
  }

  public function clear_update_transient() {
    delete_site_transient('tap_update_info_' . $this->slug);
  }
}
