<?php

namespace ThirstyAffiliates_Pro\Models\CSV_Importer;

use ThirstyAffiliates_Pro\Helpers\Plugin_Constants;
use ThirstyAffiliates_Pro\Helpers\Helper_Functions;
use ThirstyAffiliates\Helpers\Onboarding_Helper;

if ( ! class_exists( 'WP_Importer' ) ) return;  // Exit if WP_Importer doesn't exist
if ( ! defined( 'ABSPATH' ) ) exit;  // Exit if accessed directly

/**
 * Register importer class for ThirstyAffiliates Pro CSV Importer
 *
 * @since 1.0.0
 */
class CSV_Importer extends \WP_Importer {

    /*
    |--------------------------------------------------------------------------
    | Class Properties
    |--------------------------------------------------------------------------
    */

    /**
     * CSV Attachment ID.
     *
     * @since 1.0.0
     * @access private
     * @var int
     */
    private $id;

    /**
     * Model that houses all the plugin constants.
     *
     * @since 1.0.0
     * @access private
     * @var Plugin_Constants
     */
    private $_constants;

    /**
     * Property that houses all the helper functions of the plugin.
     *
     * @since 1.0.0
     * @access private
     * @var Helper_Functions
     */
    private $_helper_functions;




    /*
    |--------------------------------------------------------------------------
    | Class Methods
    |--------------------------------------------------------------------------
    */

    /**
     * Class constructor.
     *
     * @since 1.0.0
     * @access public
     *
     * @param Plugin_Constants $constants        Plugin constants object.
     * @param Helper_Functions $helper_functions Helper functions object.
     */
    public function __construct( Plugin_Constants $constants, Helper_Functions $helper_functions ) {

        $this->_constants        = $constants;
        $this->_helper_functions = $helper_functions;
    }

    /**
     * Registered handler for the CSV Importer. This function directs basically traffic depending on the step the import is up to.
     *
     * @since 1.0.0
     * @since 1.7.15 Method is no longer broken up into steps. Instead, Step 2 (importing) is handled via Ajax.
     * @access public
     */
    public function handler() {
        echo '<div class="wrap">';
        echo '<h2>' . __( 'ThirstyAffiliates CSV Importer', 'thirstyaffiliates-pro' ) . '</h2>';

        $this->introduction();

        echo '</div>'; // end .wrap
    }

    /**
     * Handles importing the CSV file through a series of batches via Ajax.
     *
     * @since 1.7.15
     * @access public
     * @return void
     */
    public function import() {
        $batch_size = apply_filters( 'tap_import_batch_size', 100 );
        $offset = isset( $_POST['offset'] ) ? (int) $_POST['offset'] : 0;
        $nonce = isset( $_POST['_wpnonce'] ) ? $_POST['_wpnonce'] : '';
        $import_key = isset( $_POST['import_key'] ) ? $_POST['import_key'] : '';
        $imported_links = array();
        $total_successful_links = 0;
        $total_failed_links = 0;
        $ta_file_id = get_transient( 'tap_import_file_id_' . $import_key );
        $links = get_transient( 'tap_import_links_' . $import_key );

        // Quick nonce verification.
        if ( ! wp_verify_nonce( $nonce, 'tap_upload_csv_form' ) ) {
            wp_send_json( array(
                'success' => false,
                'message' => esc_html__( 'There was a problem. Please try again later.', 'thirstyaffiliates-pro' )
            ) );

            exit;
        }

        // If the "tap_import_file_id" transient doesn't exist, then no CSV file has been uploaded yet.
        // In this case, upload it and store the attachment ID in the transient so it can be read from later.
        if ( ! $ta_file_id ) {
            $csv_file = $this->upload_file();

            if ( $csv_file ) {
                set_transient( 'tap_import_file_id_' . $import_key, $this->id );
            } else {
                wp_send_json( array(
                    'status'  => false,
                    'message' => esc_html__( 'File upload error.', 'thirstyaffiliates-pro' )
                ) );

                exit;
            }
        } else {
            $this->id = $ta_file_id;
        }

        // Store the parsed CSV records into a transient (if applicable) so we can work with it during each Ajax request.
        if ( ! $links ) {
            $links = $this->parse_csv();

            if ( is_array( $links ) ) {
                set_transient( 'tap_import_links_' . $import_key, $links );
            } else {
                delete_transient( 'tap_import_file_id_' . $import_key );
                wp_send_json( array(
                    'success' => false,
                    'message' => esc_html__( 'Looks like there was something wrong with the upload, the file does not exist. Please check your permissions on the wp-content/uploads directory and try again.', 'thirstyaffiliates-pro' )
                ) );

                exit;
            }
        }

        // Extract the links we're interested in based on the current offset so we can import them.
        $batched_links = array_slice( $links, $offset, $batch_size );

        foreach ( $batched_links as $link ) {
            list( $name, $url, $slug, $cats, $images, $geolinks, $autokeywords, $meta_values ) = $link;

            $result = $this->import_link( $name, $url, $slug, $cats, $images, $geolinks, $autokeywords, $meta_values );

            if ( is_string( $result ) ) {
                $imported_links[] = array(
                    'name'                     => esc_html( $name ),
                    'slug'                     => esc_html( strtolower( $slug ) ),
                    'was_successful'           => 0,
                    'error_message'            => $result
                );

                $total_failed_links++;
            } else {
                $imported_links[] = array(
                    'name'           => esc_html( $name ),
                    'slug'           => esc_html( strtolower( $slug ) ),
                    'was_successful' => 1
                );

                $total_successful_links++;
            }

            $offset++;
        }

        $count = count( $links );

        // If the offset is less than the total number of links, then we're still in the process of importing.
        if ( $offset < $count ) {
            $response = array(
                'success'                 => true,
                'offset'                  => $offset,
                'total'                   => $count,
                'imported_links'          => $imported_links,
                'total_successful_links'  => $total_successful_links,
                'total_failed_links'      => $total_failed_links
            );
        } else {
            delete_transient( 'tap_import_file_id_' . $import_key );
            delete_transient( 'tap_import_links_' . $import_key );

            $response = array(
                'success'                 => true,
                'message'                 => esc_html__( 'Import complete!', 'thirstyaffiliates-pro' ),
                'imported_links'          => $imported_links,
                'total_successful_links'  => $total_successful_links,
                'total_failed_links'      => $total_failed_links
            );
        }

        if( isset($_POST['onboarding']) && 1 === (int) $_POST['onboarding'] ) {
            Onboarding_Helper::set_has_imported_links(1);
        }

        wp_send_json( $response );
    }

    /**
     * Handles uploading of the CSV file.
     *
     * @since 1.0.0
     * @access public
     */
    private function upload_file() {

        $file = wp_import_handle_upload();

        if ( isset( $file['error'] ) ) {
            return false;
        }

        $this->id = (int) $file['id'];

        return true;
    }

    /**
     * Parses the uploaded CSV file.
     *
     * @since 1.0.0
     * @since 1.7.15 The method no longer tells the importer to import the links. Instead, this is delegated to the import() method. The method now also returns an array of parsed CSV rows.
     * @access public
     * @return array Array of parsed CSV records.
     */
    private function parse_csv() {

        set_time_limit( 0 );

        $file                = get_attached_file( $this->id );
        $first_row_flag      = true;
        $meta_keys           = array();
        $records             = array();

        if ( ( $handle = fopen( $file, "r") ) === false ) {
            return;
        }

        while ( $data = fgetcsv( $handle, 0, "," ) ) {

            $name         = trim( $data[0] );
            $url          = trim( $data[1] );
            $slug         = trim( $data[2] );
            $cats         = ! empty( $data[3] ) ? explode( ';', $data[3] ) : array();
            $images       = ! empty( $data[4] ) ? explode( ';', $data[4] ) : array();
            $geolinks     = ! empty( $data[5] ) ? explode( ';', $data[5] ) : array();
            $autokeywords = ! empty( $data[6] ) ? str_replace( ';', ',', $data[6] ) : '';

            // Store meta key
            if ( $first_row_flag ) {
                for ( $i = 7; $i < count( $data ); $i++ ) {
                    $meta_keys[ $i ] = $data[ $i ];
                }

                $first_row_flag = false;
                continue;
            }

            // 1.2: Allow user to choose if they want to escape the urls or not
            $skip_escape = isset( $_POST['skip_escape'] ) && 'true' === $_POST['skip_escape'];

            if ( ! $skip_escape ) {
                $url = esc_url_raw( $url );
            }

            if ( filter_var( $url, FILTER_VALIDATE_URL ) === false ) {
                continue;
            }

            // Get all meta values
            $meta_values = array();
            for ( $i = 7; $i < count( $data ); $i++ ) {
                $meta_values[ $meta_keys[ $i ] ] = maybe_unserialize( $data[ $i ] );
            }


            $records[] = array( $name, $url, $slug, $cats, $images, $geolinks, $autokeywords, $meta_values );
        }

        return $records;
    }

    /**
     * Given a name, url and categories list this function imports the affiliate
     *
     * @since 1.0.0
     * @access public
     *
     * @global int $user_ID Current logged in user ID.
     *
     * @param string $name         Affiliate Link post title.
     * @param string $url          Affiliate Link destination url.
     * @param string $slug         Affiliate Link post slug.
     * @param array  $cats         Affiliate Link list of categories.
     * @param array  $images       Affiliate Link attached images.
     * @param array  $geolinks     Affiliate Link list of geolocations links.
     * @param array  $autokeywords Affiliate Link list of keywords for autolinker.
     * @param array  $meta_values  Affiliate Link list of meta keys and its values.
     */
    private function import_link( $name, $url, $slug, $cats, $images, $geolinks, $autokeywords, $meta_values ) {

        global $user_ID;

        // 1.5: Allow user to override links with existing slugs
        $override_links = isset( $_POST['override_links'] ) && 'true' === $_POST['override_links'];
        $post_id        = 0;

        if ( $override_links ) {

            $post = get_posts( array(
                'name'        => $slug,
                'post_type'   => 'thirstylink',
                'post_status' => 'publish',
                'numberposts' => 1
            ) );

            if ( ! empty( $post ) && isset( $post[0] ) )
                $post_id = $post[0]->ID;
        }

        if ( $post_id ) {

            $update_post = array(
                'ID'           => $post_id,
                'post_title'   => $name,
                'post_content' => '',
                'post_status'  => 'publish',
            );

            $post_id = wp_update_post( $update_post );
        }

        if ( is_wp_error( $post_id ) ) {
            return $post_id->get_error_message();
        }

        $thirstylink = ThirstyAffiliates()->helpers['Helper_Functions']->get_affiliate_link( $post_id );

        /* unset categories and images when $ovverideLinks is true */
        if ( $override_links ) {

            // unassign all thirstylink-category terms from affiliate link
            $this->_unassign_all_terms_from_link( $post_id );
        }

        // set properties
        $thirstylink->set_prop( 'name', sanitize_text_field( $name ) );
        $thirstylink->set_prop( 'slug', sanitize_text_field( $slug ) );
        $thirstylink->set_prop( 'destination_url', $url );

        // process all meta keys that are under ThirstyAffiliates (_ta_)
        foreach ( $meta_values as $meta_key => $meta_value ) {

            if ( substr( $meta_key, 0, 4 ) !== Plugin_Constants::META_DATA_PREFIX )
                continue;

            $prop_key  = str_replace( Plugin_Constants::META_DATA_PREFIX, '', $meta_key );
            $thirstylink->set_prop( $prop_key, maybe_unserialize( $meta_value ) );
            unset( $meta_values[ $meta_key ] );
        }

        $is_saved = $thirstylink->save();
        $this->process_categories( $thirstylink->get_id(), $cats );

        update_post_meta( $thirstylink->get_id(), Plugin_Constants::META_DATA_PREFIX . 'image_ids', $this->process_images( $post_id, $images, $name ) );
        update_post_meta( $thirstylink->get_id(), Plugin_Constants::META_DATA_PREFIX . 'geolocation_links', $this->process_geolocation( $geolinks ) );
        update_post_meta( $thirstylink->get_id(), Plugin_Constants::META_DATA_PREFIX . 'autolink_keyword_list', sanitize_text_field( $autokeywords ) );

        // process meta keys that are not for ThirstyAffiliates
        foreach ( $meta_values as $meta_key => $meta_value ) {

            $meta_value = gettype( $meta_value ) == 'array' ? maybe_unserialize( $meta_value ) : sanitize_text_field( $meta_value );
            update_post_meta( $thirstylink->get_id(), $meta_key, $meta_value );
        }

        return true;
    }

    /**
     * Process images and set them to the Affiliate Link post.
     *
     * @since 1.0.0
     * @access private
     *
     * @param int    $post_id Affiliate Link post ID.
     * @param array  $images  List of image urls to be processed and attached.
     * @param string $name    Affiiate Link post title.
     */
    private function process_images( $post_id, $images, $name ) {

        if ( ! is_array( $images ) || empty( $images ) )
            return;

        $override_links = isset( $_POST['override_links'] ) && 'true' === $_POST['override_links'];
        $attachments    = ( $post_id ) ? get_post_meta( $post_id, Plugin_Constants::META_DATA_PREFIX . 'image_ids', true ) : array();

        if ( ! is_array( $attachments ) )
            $attachments = array();

        foreach ( $images as $key => $img_url ) {

            $file          = array();
            $img_url       = trim( $img_url );
            $attachment_id = $this->_get_attachment_id_by_url( $img_url );

            if ( ! $img_url )
                continue;

            if ( in_array( $attachment_id, $attachments ) ) {
                continue;
            }

            $tmp_img = download_url( $img_url );

            // If error storing temporarily, unlink
            if ( is_wp_error( $tmp_img ) ) {

                @unlink( $file['tmp_name'] );
                $file['tmp_name'] = '';
                continue;
            }

            // Set variables for storage. Fix filename for query strings
            preg_match( '/[^\?]+\.(jpg|jpe|jpeg|gif|png)/i', $img_url, $matches );
            $file['name']     = basename( $matches[0] );
            $file['tmp_name'] = $tmp_img;

            // do the validation and storage stuff
            $img_id = media_handle_sideload( $file, $post_id, $name );

            if ( is_wp_error( $img_id ) || ! $img_id) {
                @unlink( $file['tmp_name'] );
                continue;
            }

            // assign image to post
            $attachments[] = $img_id;
        }

        return array_unique( $attachments );
    }

    /**
     * Process categories and set the terms to the Affiliate Link post.
     *
     * @since 1.0.0
     * @access private
     *
     * @param int   $post_id   Affiliate Link post ID.
     * @param array $cat_names List of category names.
     */
    private function process_categories( $post_id, $cat_names ) {

        $terms = array();

        foreach ( $cat_names as $cat_name ) {

            $cat_slug = sanitize_title_with_dashes( trim( $cat_name ) );
            $term     = term_exists( $cat_slug, Plugin_Constants::AFFILIATE_LINKS_TAX );
            $term_id  = is_array( $term ) && isset( $term['term_id'] ) ? (int) $term['term_id'] : 0;

            if ( ! $term_id ) {
                $term = wp_insert_term( $cat_name, Plugin_Constants::AFFILIATE_LINKS_TAX, array(
                    'parent'      => 0,
                    'description' => '',
                ) );

                $term_id = ! is_wp_error( $term ) ? (int) $term['term_id'] : 0;
            }

            if ( $term_id && gettype( $term_id ) == 'integer' )
                $terms[] = $term_id;
        }

        wp_set_object_terms( $post_id, $terms, Plugin_Constants::AFFILIATE_LINKS_TAX, true );
    }

    /**
     * Process images and set them to the Affiliate Link post.
     *
     * @since 1.0.0
     * @access private
     *
     * @param array $images  List of image urls to be processed and attached.
     * @return array Converted geolinks from old to new format.
     */
    private function process_geolocation( $csv_raw_geolinks ) {

        $old_geolinks = array();

        if ( ! is_array( $csv_raw_geolinks ) || empty( $csv_raw_geolinks ) )
            return $old_geolinks;

        foreach ( $csv_raw_geolinks as $raw_geolink ) {
            $temp = explode( ':', $raw_geolink, 2 );
            $old_geolinks[ $temp[0] ] = $temp[1];
        }

        return $this->_helper_functions->convert_geolinks_old_to_new_format( $old_geolinks );
    }

    /**
     * Output the introductory text and the importer form
     *
     * @since 1.0.0
     * @access private
     */
    private function introduction() {
        $bytes          = apply_filters( 'import_upload_size_limit', wp_max_upload_size() );
        $max_size       = size_format( $bytes );
        $upload_dir     = wp_upload_dir();
        $csv_sample_url = $this->_constants->PLUGIN_DIR_URL() . 'sample.csv';
        $country_list   = $this->_constants->PLUGIN_DIR_URL() . 'countryList.xml';

        include_once( $this->_constants->VIEWS_ROOT_PATH() . 'csv-importer/view-csv-importer.php' );
    }

    /**
     * Removes all thirstylink-category assigned to a affiliate link
     *
     * @since 1.0.0
     * @access private
     *
     * @global wpdb $wpdb Object that contains a set of functions used to interact with a database.
     *
     * @param int $post_id Affiliate Link post ID.
     */
    private function _unassign_all_terms_from_link( $post_id ) {

        global $wpdb;

        $terms_db  = $wpdb->get_results( "SELECT `term_id` FROM $wpdb->term_taxonomy WHERE `taxonomy` = 'thirstylink-category'", ARRAY_N );
        $all_terms = array_reduce( $terms_db, 'array_merge', array() );
        $all_terms = array_map( 'intval', $all_terms );

        wp_remove_object_terms( $post_id, $all_terms, 'thirstylink-category' );
    }

    /**
     * Get attachment ID by image URL.
     *
     * @since 1.0.0
     * @access private
     *
     * @param string $image_url URL of image to process.
     * @return int Attachment ID.
     */
    private function _get_attachment_id_by_url( $image_url ) {

        global $wpdb;

        if ( ! $image_url )
            return;

        return $wpdb->get_var( $wpdb->prepare( "SELECT ID FROM $wpdb->posts WHERE guid='%s';", $image_url ) );
    }
}
