<?php
// Register Custom Post Type: Properties
function custom_cpt_properties() {
    $labels = array(
        'name'               => __('Properties', 'custom'),
        'singular_name'      => __('Property', 'custom'),
        'menu_name'          => __('Properties', 'custom'),
        'add_new'            => __('Add Property', 'custom'),
        'add_new_item'       => __('Add New Property', 'custom'),
        'edit_item'          => __('Edit Property', 'custom'),
        'new_item'           => __('New Property', 'custom'),
        'view_item'          => __('View Property', 'custom'),
        'search_items'       => __('Search Properties', 'custom'),
        'not_found'          => __('No Properties found', 'custom'),
        'not_found_in_trash' => __('No Properties found in Trash', 'custom'),
    );

    $args = array(
        'labels'             => $labels,
        'public'             => true,
        'has_archive'        => true,
        'menu_icon'          => 'dashicons-building', // Icon in admin menu
        // 'supports'           => array('title', 'editor', 'thumbnail', 'custom-fields'),
        'supports' => array('title', 'editor', 'thumbnail', 'custom-fields'),

        'rewrite'            => array('slug' => 'properties'),
        'show_in_rest'       => true, // For Gutenberg support
    );

    register_post_type('properties', $args);
}
add_action('init', 'custom_cpt_properties');
add_action('after_setup_theme', 'properties_cpt_support_thumbnail');
add_theme_support('post-thumbnails');


// Register Taxonomy: Property Type
function custom_taxonomy_property_type() {
    $labels = array(
        'name'              => __('Property Types', 'custom'),
        'singular_name'     => __('Property Type', 'custom'),
        'search_items'      => __('Search Property Types', 'custom'),
        'all_items'         => __('All Property Types', 'custom'),
        'edit_item'         => __('Edit Property Type', 'custom'),
        'add_new_item'      => __('Add New Property Type', 'custom'),
    );

    $args = array(
        'labels'            => $labels,
        'hierarchical'      => true,
        'public'            => true,
        'rewrite'           => array('slug' => 'property-type'),
        'show_in_rest'      => true, // For Gutenberg
    );

    register_taxonomy('property_type', 'properties', $args);
}
add_action('init', 'custom_taxonomy_property_type');


// Enable Featured Image for Properties CPT
function properties_cpt_support_thumbnail() {
    add_post_type_support('properties', 'thumbnail');
}
add_action('init', 'properties_cpt_support_thumbnail');

//===============
// Filter in Admin for Property Type & Location
function custom_filter_properties_by_taxonomy($post_type) {
    global $typenow;

    if ($typenow == 'properties') {
        $taxonomies = array('property_type');

        foreach ($taxonomies as $taxonomy) {
            $taxonomy_obj = get_taxonomy($taxonomy);
            $terms = get_terms(array(
                'taxonomy' => $taxonomy,
                'hide_empty' => false,
            ));

            if ($terms) {
                echo '<select name="' . $taxonomy . '" id="' . $taxonomy . '" class="postform">';
                echo '<option value="">' . sprintf(__('All %s', 'custom'), $taxonomy_obj->labels->name) . '</option>';

                foreach ($terms as $term) {
                    $selected = (isset($_GET[$taxonomy]) && $_GET[$taxonomy] == $term->slug) ? ' selected="selected"' : '';
                    echo '<option value="' . $term->slug . '"' . $selected . '>' . $term->name . '</option>';
                }
                echo '</select>';
            }
        }
    }
}
add_action('restrict_manage_posts', 'custom_filter_properties_by_taxonomy');

// Filter Query by Selected Taxonomy
function custom_filter_query_by_taxonomy($query) {
    global $pagenow;
    $post_type = 'properties';
    $taxonomies = array('property_type');

    if ($pagenow == 'edit.php' && isset($_GET['post_type']) && $_GET['post_type'] == $post_type) {
        foreach ($taxonomies as $taxonomy) {
            if (isset($_GET[$taxonomy]) && $_GET[$taxonomy] != '') {
                $query->query_vars[$taxonomy] = $_GET[$taxonomy];
            }
        }
    }
}
add_filter('parse_query', 'custom_filter_query_by_taxonomy');
//==========================

// Add Sorting Functionality
function custom_properties_orderby($query) {
    if (!is_admin() || !$query->is_main_query()) {
        return;
    }

    if ($query->get('orderby') == 'property_type') {
        $query->set('orderby', 'taxonomy');
        $query->set('tax_query', array(
            array(
                'taxonomy' => 'property_type',
                'field' => 'slug',
                'terms' => '',
            ),
        ));
    }


}
add_action('pre_get_posts', 'custom_properties_orderby'); 
//=============
 // Manage Custom Columns for Properties CPT
function custom_properties_columns($columns) {
    unset($columns['date']); // Remove default Date column
    $columns['thumbnail'] = __('Image', 'custom'); // Image Column
    $columns['property_type'] = __('Property Type', 'custom'); // Property Type Column
    $columns['property_price'] = __('Price', 'custom'); // Price Column
    $columns['date'] = __('Date', 'custom'); // Re-add Date Column
    return $columns;
}
add_filter('manage_properties_posts_columns', 'custom_properties_columns');

// Display Custom Column Content
function custom_properties_columns_content($column, $post_id) {
    switch ($column) {
        case 'thumbnail':
            if (has_post_thumbnail($post_id)) {
                echo get_the_post_thumbnail($post_id, 'thumbnail', ['style' => 'width: 60px; height: auto; border-radius: 5px;']);
            } else {
                echo __('No Image', 'custom');
            }
            break;

        case 'property_type':
            $terms = get_the_terms($post_id, 'property_type');
            if (!empty($terms)) {
                $term_names = wp_list_pluck($terms, 'name');
                echo implode(', ', $term_names);
            } else {
                echo '—';
            }
            break;

        case 'property_price':
            $terms = get_the_terms($post_id, 'property_type');
            if (!empty($terms)) {
                foreach ($terms as $term) {
                    $price = get_term_meta($term->term_id, '_property_final_price', true);
                    if (!empty($price)) {
                        echo esc_html($price);
                        break; // Show only the first matched price
                    }
                }
            } else {
                echo '—';
            }
            break;
    }
}
add_action('manage_properties_posts_custom_column', 'custom_properties_columns_content', 10, 2);

// Make Thumbnail Column Sortable
function properties_sortable_columns($columns) {
    $columns['property_price'] = 'property_price';
    return $columns;
}
add_filter('manage_edit-properties_sortable_columns', 'properties_sortable_columns');

//===========
// Add Meta Fields to All Taxonomies of CPT
function add_meta_fields_to_taxonomies($term) {
    ?>
    <div class="form-field">
        <label for="_property_price"><?php _e('Price', 'custom'); ?></label>
        <input type="text" name="_property_price" id="_property_price" value="">
    </div>
    <div class="form-field">
        <label for="_property_discount"><?php _e('Discount (%)', 'custom'); ?></label>
        <input type="text" name="_property_discount" id="_property_discount" value="">
    </div>
    <div class="form-field">
        <label for="_property_tax"><?php _e('Tax (%)', 'custom'); ?></label>
        <input type="text" name="_property_tax" id="_property_tax" value="">
    </div>
    <div class="form-field">
        <label for="_property_final_price"><?php _e('Final Price', 'custom'); ?></label>
        <input type="text" name="_property_final_price" id="_property_final_price" value="" readonly>
    </div>
    <?php
}
add_action('property_type_add_form_fields', 'add_meta_fields_to_taxonomies', 10, 1);

function save_taxonomy_meta_fields($term_id) {
    $price = isset($_POST['_property_price']) ? sanitize_text_field($_POST['_property_price']) : 0;
    $discount = isset($_POST['_property_discount']) ? sanitize_text_field($_POST['_property_discount']) : 0;
    $tax = isset($_POST['_property_tax']) ? sanitize_text_field($_POST['_property_tax']) : 0;

    update_term_meta($term_id, '_property_price', $price);
    update_term_meta($term_id, '_property_discount', $discount);
    update_term_meta($term_id, '_property_tax', $tax);

    // Calculation with default 0 if fields are empty
    $price = !empty($price) ? floatval($price) : 0;
    $discount = !empty($discount) ? floatval($discount) : 0;
    $tax = !empty($tax) ? floatval($tax) : 0;

    $discounted_price = $price - ($price * ($discount / 100));
    $final_price = $discounted_price + ($discounted_price * ($tax / 100));

    update_term_meta($term_id, '_property_final_price', $final_price);
}
add_action('created_property_type', 'save_taxonomy_meta_fields', 10, 2);
add_action('edited_property_type', 'save_taxonomy_meta_fields', 10, 2);

function edit_meta_fields_taxonomies($term) {
    $price = get_term_meta($term->term_id, '_property_price', true);
    $discount = get_term_meta($term->term_id, '_property_discount', true);
    $tax = get_term_meta($term->term_id, '_property_tax', true);
    $final_price = get_term_meta($term->term_id, '_property_final_price', true);
    ?>
    <tr class="form-field">
        <th scope="row" valign="top"><label for="_property_price"><?php _e('Price', 'custom'); ?></label></th>
        <td><input type="text" name="_property_price" id="_property_price" value="<?php echo esc_attr($price); ?>"></td>
    </tr>
    <tr class="form-field">
        <th scope="row" valign="top"><label for="_property_discount"><?php _e('Discount (%)', 'custom'); ?></label></th>
        <td><input type="text" name="_property_discount" id="_property_discount" value="<?php echo esc_attr($discount); ?>"></td>
    </tr>
    <tr class="form-field">
        <th scope="row" valign="top"><label for="_property_tax"><?php _e('Tax (%)', 'custom'); ?></label></th>
        <td><input type="text" name="_property_tax" id="_property_tax" value="<?php echo esc_attr($tax); ?>"></td>
    </tr>
    <tr class="form-field">
        <th scope="row" valign="top"><label for="_property_final_price"><?php _e('Final Price', 'custom'); ?></label></th>
        <td><input type="text" name="_property_final_price" id="_property_final_price" value="<?php echo esc_attr($final_price); ?>" readonly></td>
    </tr>
    <?php
}
add_action('property_type_edit_form_fields', 'edit_meta_fields_taxonomies', 10, 2);
